% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_gbifid.R
\name{get_gbifid}
\alias{get_gbifid}
\alias{as.gbifid}
\alias{as.gbifid.gbifid}
\alias{as.gbifid.character}
\alias{as.gbifid.list}
\alias{as.gbifid.numeric}
\alias{as.gbifid.data.frame}
\alias{as.data.frame.gbifid}
\alias{get_gbifid_}
\title{Get the GBIF backbone taxon ID from taxonomic names.}
\usage{
get_gbifid(sciname, ask = TRUE, messages = TRUE, rows = NA,
  phylum = NULL, class = NULL, order = NULL, family = NULL,
  rank = NULL, method = "backbone", ...)

as.gbifid(x, check = FALSE)

\method{as.gbifid}{gbifid}(x, check = FALSE)

\method{as.gbifid}{character}(x, check = TRUE)

\method{as.gbifid}{list}(x, check = TRUE)

\method{as.gbifid}{numeric}(x, check = TRUE)

\method{as.gbifid}{data.frame}(x, check = TRUE)

\method{as.data.frame}{gbifid}(x, ...)

get_gbifid_(sciname, messages = TRUE, rows = NA, method = "backbone")
}
\arguments{
\item{sciname}{character; scientific name.}

\item{ask}{logical; should get_colid be run in interactive mode?
If TRUE and more than one ID is found for the species, the user is asked for
input. If FALSE NA is returned for multiple matches.}

\item{messages}{logical; If TRUE the actual taxon queried is printed on the console.}

\item{rows}{numeric; Any number from 1 to infinity. If the default NA, all rows are considered.
Note that this function still only gives back a gbifid class object with one to many identifiers.
See \code{\link[taxize]{get_gbifid_}} to get back all, or a subset, of the raw data that you are
presented during the ask process.}

\item{phylum}{(character) A phylum (aka division) name. Optional. See \code{Filtering}
below.}

\item{class}{(character) A class name. Optional. See \code{Filtering} below.}

\item{order}{(character) An order name. Optional. See \code{Filtering} below.}

\item{family}{(character) A family name. Optional. See \code{Filtering} below.}

\item{rank}{(character) A taxonomic rank name. See \code{\link{rank_ref}} for possible
options. Though note that some data sources use atypical ranks, so inspect the
data itself for options. Optional. See \code{Filtering} below.}

\item{method}{(character) one of "backbone" or "lookup". See Details.}

\item{...}{Ignored}

\item{x}{Input to \code{\link{as.gbifid}}}

\item{check}{logical; Check if ID matches any existing on the DB, only used in
\code{\link{as.gbifid}}}
}
\value{
A vector of taxonomic identifiers as an S3 class.
If a taxon is not found an \code{NA} is given. If more than one identifier
is found the function asks for user input if \code{ask = TRUE}, otherwise
returns \code{NA}. If \code{ask=FALSE} and \code{rows} does not equal
\code{NA}, then a data.frame is given back, but not of the uid class, which
you can't pass on to other functions as you normally can.

See \code{\link{get_id_details}} for further details including
attributes and exceptions
}
\description{
Get the GBIF backbone taxon ID from taxonomic names.
}
\details{
Internally in this function we use a function to search GBIF's taxonomy,
and if we find an exact match we return the ID for that match. If there isn't an
exact match we return the options to you to pick from.
}
\section{method parameter}{

"backbone" uses the \code{/species/match} GBIF API route, matching against their
backbone taxonomy. We turn on fuzzy matching by default, as the search without
fuzzy against backbone is quite narrow. "lookup" uses the \code{/species/search}
GBIF API route, doing a full text search of name usages covering scientific
and vernacular named, species descriptions, distributions and the entire
classification.
}

\section{Filtering}{

The parameters \code{phylum}, \code{class}, \code{order}, \code{family}, and \code{rank}
are not used in the search to the data provider, but are used in filtering the data down
to a subset that is closer to the target you want.  For all these parameters,
you can use regex strings since we use \code{\link{grep}} internally to match.
Filtering narrows down to the set that matches your query, and removes the rest.
}

\examples{
\dontrun{
get_gbifid(sciname='Poa annua')
get_gbifid(sciname='Pinus contorta')
get_gbifid(sciname='Puma concolor')

# multiple names
get_gbifid(c("Poa annua", "Pinus contorta"))

# specify rows to limit choices available
get_gbifid(sciname='Pinus')
get_gbifid(sciname='Pinus', rows=10)
get_gbifid(sciname='Pinus', rows=1:3)

# When not found, NA given
get_gbifid(sciname="uaudnadndj")
get_gbifid(c("Chironomus riparius", "uaudnadndj"))

# Narrow down results to a division or rank, or both
## Satyrium example
### Results w/o narrowing
get_gbifid("Satyrium")
### w/ phylum
get_gbifid("Satyrium", phylum = "Tracheophyta")
get_gbifid("Satyrium", phylum = "Arthropoda")
### w/ phylum & rank
get_gbifid("Satyrium", phylum = "Arthropoda", rank = "genus")

## Rank example
get_gbifid("Poa", method = "lookup")
get_gbifid("Poa", method = "lookup", rank = "genus")
get_gbifid("Poa", method = "lookup", family = "Thripidae")

# Fuzzy filter on any filtering fields
## uses grep on the inside
get_gbifid("Satyrium", phylum = "arthropoda")
get_gbifid("A*", method = "lookup", order = "*tera")
get_gbifid("A*", method = "lookup", order = "*ales")

# Convert a uid without class information to a uid class
as.gbifid(get_gbifid("Poa annua")) # already a uid, returns the same
as.gbifid(get_gbifid(c("Poa annua","Puma concolor"))) # same
as.gbifid(2704179) # numeric
as.gbifid(c(2704179,2435099,3171445)) # numeric vector, length > 1
as.gbifid("2704179") # character
as.gbifid(c("2704179","2435099","3171445")) # character vector, length > 1
as.gbifid(list("2704179","2435099","3171445")) # list, either numeric or character
## dont check, much faster
as.gbifid("2704179", check=FALSE)
as.gbifid(2704179, check=FALSE)
as.gbifid(2704179, check=FALSE)
as.gbifid(c("2704179","2435099","3171445"), check=FALSE)
as.gbifid(list("2704179","2435099","3171445"), check=FALSE)

(out <- as.gbifid(c(2704179,2435099,3171445)))
data.frame(out)
as.uid( data.frame(out) )

# Get all data back
get_gbifid_("Puma concolor")
get_gbifid_(c("Pinus", "uaudnadndj"))
get_gbifid_(c("Pinus", "Puma"), rows=5)
get_gbifid_(c("Pinus", "Puma"), rows=1:5)

# use curl options
invisible(get_gbifid("Quercus douglasii", verbose = TRUE))
}
}
\seealso{
\code{\link[taxize]{classification}}

Other taxonomic-ids: \code{\link{get_boldid}},
  \code{\link{get_colid}}, \code{\link{get_eolid}},
  \code{\link{get_ids}}, \code{\link{get_iucn}},
  \code{\link{get_natservid}}, \code{\link{get_nbnid}},
  \code{\link{get_tolid}}, \code{\link{get_tpsid}},
  \code{\link{get_tsn}}, \code{\link{get_uid}},
  \code{\link{get_wiki}}, \code{\link{get_wormsid}}
}
\author{
Scott Chamberlain, \email{myrmecocystus@gmail.com}
}
\concept{taxonomic-ids}
