/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __IF_STAT_HPP_INCLUDED
#define __IF_STAT_HPP_INCLUDED

#include "frontend/ast/ConditionedStat.hpp"
#include "frontend/ast/Expression.hpp"

namespace ast {

/** an if-then-else statement */
class IfStat : public ConditionedStat {
public:
	//! c'tor
	/** @param cond condition that needs to be evaluated to boolean
          * @param thenList list of statements to be executed if cond is true
          * @param elseList list of statements to be executed if cond is not
          *        met.
	  * @param loc location of the declaration.
          */
	IfStat(
		Expression *cond, 
		std::list<SeqStat*> *thenList, 
		std::list<SeqStat*> *elseList,
		Location loc
		) : 	ConditionedStat(cond, loc), 
			thenStats(thenList), 
			elseStats(elseList) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << "IF " << this->condition << " THEN ";
		util::MiscUtil::listPut(this->thenStats, stream, "; ");
		if (this->elseStats) {
			stream << " ELSE ";
			util::MiscUtil::listPut(this->elseStats, stream, 
						"; ");
		}
		stream << " END IF;";
	}

	
	/** sequential statements that are part of the then-clause */
	std::list<SeqStat*> *thenStats;

	/** sequential statements that are part of the else-clause */
	std::list<SeqStat*> *elseStats;

protected:
	/** Destructor */
	virtual ~IfStat() {
		util::MiscUtil::lterminate(thenStats);
		util::MiscUtil::lterminate(elseStats);
	}
};

}; /* namespace ast */

#endif /* __IF_STAT_HPP_INCLUDED */
