/**
 * Simple audio playback object. Each object loads and stores a single WAV sound
 * file. WAV files must use the PCM encoding with no compression and 8, 16, 24
 * or 32 bits per sample, any number of channels.
 * 
 * Features:
 * 
 * - Start and stop playback.
 * 
 * - One-shot or loop playback.
 * 
 * - Cursor positioning.
 * 
 * - Change playback frequency rate.
 * 
 * Example that loads a sound, plays the sound and waits for its ending:
 * 
 * <pre>
 * audio_Type *a = audio_new("MySong.wav");
 * audio_play(a);
 * while( audio_isPlaying(a) )  Sleep(1000);
 * memory_dispose(a);
 * </pre>
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2017/09/06 16:36:45 $
 */

#ifndef AUDIO_H
#define AUDIO_H

#ifdef audio_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Playable sound.
 */
typedef struct audio_Type audio_Type;

/**
 * Create new WAV file player. The player is paused, positioned at the beginning
 * of the sound, in one-shot mode and at its nominal sampling rate.
 * @param wav_filename WAV file to play.
 * @return New WAV file player. Can be released with memory_dispose().
 */
EXTERN audio_Type * audio_new(char * wav_filename);

/**
 * Starts or continue playing up to end of the sound (or forever if loop
 * enabled).
 */
EXTERN void audio_play(audio_Type *this);

/**
 * Tells if currently playing the sound.
 * @return True if currently playing the sound; false if either end of the
 * sound and no loop enabled, or paused.
 */
EXTERN int  audio_isPlaying(audio_Type *this);

/** Puts sound in pause status. */
EXTERN void audio_pause(audio_Type *this);

/** Tells if the sound is currently paused. */
EXTERN int  audio_isPaused(audio_Type *this);

/** Enables or disables the loop mode. */
EXTERN void audio_loop(audio_Type *this, int enable);

/** Tells if loop mode is enabled. */
EXTERN int  audio_isLooping(audio_Type *this);

/** Returns the original total length of the sound (ms). */
EXTERN int  audio_getOriginalLengthMilliseconds(audio_Type *this);

/**
 * Returns the current total length of the sound based on the current playback
 * rate (ms).
 */
EXTERN int  audio_getCurrentLengthMilliseconds(audio_Type *this);

/**
 * Returns the current position of the cursor based on the current playback
 * rate (ms).
 */
EXTERN int  audio_getCurrentPositionMilliseconds(audio_Type *this);

/**
 * Moves the cursor at the specified time (ms) based on the current playback
 * rate. Negative values moves to the beginning. Values longer than the total
 * sound length move the cursor at the end. Sound playback continues based on
 * the current setting of the loop and pause modes.
 */
EXTERN void audio_setCurrentPositionMilliseconds(audio_Type *this, int position);

/** Returns the original samples playback rate of the sound (Hz). */
EXTERN int  audio_getOriginalSamplesPerSecond(audio_Type *this);

/** Returns the current samples playback rate of the sound (Hz). */
EXTERN int  audio_getCurrentSamplesPerSecond(audio_Type *this);

/**
 * Set the current samples playback rate (Hz). The specified value is forced to
 * the range [100,15000] Hz.
 */
EXTERN void audio_setCurrentSamplesPerSecond(audio_Type *this, int rate);

#undef EXTERN
#endif

