/*
 *  acm : an aerial combat simulator for X
 *  Copyright (C) 1991-1998  Riley Rainey
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

#ifndef _weapon_h
#define _weapon_h

#include "pm.h"

#ifdef weapon_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/* Index of the weapon into the wtbl[] table: */
#define weapon_NULL     0     /* null weapon, or no weapon selected */
#define weapon_M61A1    1     /* M-61A1 Vulcan 20mm cannon */
#define weapon_AIM9M    2     /* Sidewinder missile */
#define weapon_AIM120   3     /* AIM-120 missile */
#define weapon_MK82     4     /* Mark 82 (510 lb) gravity bomb */
//#define weapon_AGM65    5      /* FIXME: not used */

/**
 * Defines call-back procedures for all the standard actions of a weapon.
 * In other words, this type defines the "abstract class" from which any other
 * specific weapon is derived.
 */
typedef struct
{
	/**
	 * Weapon select procedure invoked by pilot.
	 * A selection function normally determines whether there are any weapons
	 * of this type on-board.  If so, and the weapon system is functional
	 * (in other words, undamaged) then return true, otherwise return 0.
	 * Return: true if there is actually a weapon of this type available and
	 * functional.
	 */
	int  (*select) (struct craft *);

	/**
	 * Per tick update procedure that implements weapon dynamics.
	 * Return: FIXME.
	 */
	int  (*update) (struct craft *);

	/**
	 * Display (that is HUD) update procedure.
	 * Return: TRUE if there is a reasonable chance of scoring a kill.
	 */
	int  (*display) (struct craft *, craftType *, viewer *, int, int);

	/**
	 * Fire button pressed procedure.
	 * Return: FIXME.
	 */
	int  (*firePress) (struct craft *);

	/**
	 * Fire button released.
	 * Return: FIXME.
	 */
	int  (*fireRelease) (struct craft *);

	/**
	 * Description of this weapon.
	 */
	craftType *w;

} weapon_Type;


/**
 * Initialize the module.
 */
EXTERN void weapon_init(void);

/**
 * Return the weapon number WK_* given its name.
 * Return a negative value if name unknown.
 */
EXTERN int weapon_nameToId(char * name);

/**
 * Return the weapon name given its id WK_*.
 * Return NULL if the id is invalid.
 */
EXTERN char * weapon_idToName(int id);

/**
 * Count a particular type of ordinance on an aircraft, where id is
 * one of the constants WK_* (see pm.h).
 */
EXTERN int weapon_countOrdinance(craft * c, int id);

/**
 *  Draw HUD data specific of the currently selected weapon.
 *  fpm_x,fpm_y is the current position of the HUD FPM (used by the
 *  ccip module).
 *  Return a nonzero value if have a reasonable chance of scoring a kill.
 */
EXTERN int weapon_displaySelected(craft * c, viewer * u, int fpm_x, int fpm_y);

EXTERN int weapon_update(craft * c);

/**
 * Fire the weapon currently selected.
 * Returns 0 if no weapon is currently available.
 */
EXTERN int weapon_fire(craft * c);

EXTERN int weapon_ceaseFire(craft * c);

/**
 *  Return the index of the first station with a particular type of ordinance
 *  ready for fire.
 *  id is one of the constants WK_* (see pm.h).
 */
EXTERN int weapon_getReadyStation(craft * c, int id);

/**
 * Select the given weapon number id (see constants WK_* in pm.h).
 * Returns 0 if the selection cannot be operated.
 */
EXTERN int weapon_selectByName(craft * c, int id);

/**
 * Select next available type of weapon.
 * Returns 0 if the selection cannot be operated.
 */
EXTERN int weapon_selectNextAvailable(craft * c);

#undef EXTERN
#endif
